/*
 * Decompiled with CFR 0.152.
 */
package de.rcenvironment.core.authentication.internal;

import de.rcenvironment.core.authentication.AuthenticationException;
import de.rcenvironment.core.authentication.AuthenticationService;
import de.rcenvironment.core.authentication.CertificateUser;
import de.rcenvironment.core.authentication.LDAPUser;
import de.rcenvironment.core.authentication.SingleUser;
import de.rcenvironment.core.authentication.User;
import de.rcenvironment.core.authentication.internal.AuthenticationConfiguration;
import de.rcenvironment.core.configuration.ConfigurationService;
import de.rcenvironment.core.utils.common.StringUtils;
import de.rcenvironment.core.utils.incubator.Assertions;
import java.io.IOException;
import java.security.GeneralSecurityException;
import java.security.InvalidKeyException;
import java.security.NoSuchAlgorithmException;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.cert.CertificateExpiredException;
import java.security.cert.CertificateNotYetValidException;
import java.security.cert.X509CRL;
import java.security.cert.X509Certificate;
import java.util.Date;
import java.util.List;
import java.util.Properties;
import java.util.Random;
import java.util.Vector;
import javax.crypto.BadPaddingException;
import javax.crypto.Cipher;
import javax.crypto.IllegalBlockSizeException;
import javax.crypto.NoSuchPaddingException;
import javax.naming.NamingException;
import javax.naming.directory.InitialDirContext;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.globus.gsi.CertUtil;
import org.globus.gsi.OpenSSLKey;
import org.globus.gsi.bc.BouncyCastleOpenSSLKey;
import org.osgi.framework.BundleContext;

public class AuthenticationServiceImpl
implements AuthenticationService {
    private static final String REFERRAL = "follow";
    private static final String CONTEXT_FACTORY_CLASS = "com.sun.jndi.ldap.LdapCtxFactory";
    private static final String LDAP_AUTH_METHOD = "simple";
    private static final String LDAP_PROTOCOL = "ldap://";
    private static final String ASSERTIONS_PARAMETER_NULL = "The parameter \"%s\" must not be null.";
    private static final String DSA = "DSA";
    private static final String RSA = "RSA";
    private static final Log LOGGER = LogFactory.getLog(AuthenticationServiceImpl.class);
    private static final String CERTIFICATE_COULD_NOT_BE_LOADED = "The given CA certificate (%s) could not be loaded.";
    private static final String CRL_COULD_NOT_BE_LOADED = "The given certificate revocation list (CRL) (%s) could not be loaded.";
    private AuthenticationConfiguration myConfiguration;
    private ConfigurationService configurationService;
    private String bundleSymbolicName;

    protected void activate(BundleContext context) {
        this.bundleSymbolicName = context.getBundle().getSymbolicName();
        this.myConfiguration = new AuthenticationConfiguration();
    }

    protected void bindConfigurationService(ConfigurationService newConfigurationService) {
        this.configurationService = newConfigurationService;
    }

    @Override
    @Deprecated
    public AuthenticationService.X509AuthenticationResult authenticate(X509Certificate certificate, OpenSSLKey encryptedKey, String password) throws AuthenticationException {
        Assertions.isDefined((Object)certificate, (String)StringUtils.format((String)ASSERTIONS_PARAMETER_NULL, (Object[])new Object[]{"certificate"}));
        Assertions.isDefined((Object)encryptedKey, (String)StringUtils.format((String)ASSERTIONS_PARAMETER_NULL, (Object[])new Object[]{"key"}));
        AuthenticationService.X509AuthenticationResult result = null;
        try {
            certificate.checkValidity();
        }
        catch (CertificateNotYetValidException e) {
            throw new AuthenticationException(e);
        }
        catch (CertificateExpiredException e) {
            throw new AuthenticationException(e);
        }
        if (password == null && this.isPasswordNeeded(encryptedKey)) {
            result = AuthenticationService.X509AuthenticationResult.PASSWORD_REQUIRED;
        }
        if (result == null && !this.isPasswordCorrect(encryptedKey, password)) {
            result = AuthenticationService.X509AuthenticationResult.PASSWORD_INCORRECT;
        }
        if (result == null && !this.isPrivateKeyBelongingToCertificate(certificate, encryptedKey)) {
            result = AuthenticationService.X509AuthenticationResult.PRIVATE_KEY_NOT_BELONGS_TO_PUBLIC_KEY;
        }
        if (result == null && !this.isSignedByTrustedCA(certificate)) {
            result = AuthenticationService.X509AuthenticationResult.NOT_SIGNED_BY_TRUSTED_CA;
        }
        if (result == null && this.isRevoced(certificate)) {
            result = AuthenticationService.X509AuthenticationResult.CERTIFICATE_REVOKED;
        }
        if (result == null) {
            result = AuthenticationService.X509AuthenticationResult.AUTHENTICATED;
        }
        return result;
    }

    @Override
    public AuthenticationService.LDAPAuthenticationResult authenticate(String username, String password) {
        if (password == null || password.trim().isEmpty() || username == null || username.trim().isEmpty()) {
            return AuthenticationService.LDAPAuthenticationResult.PASSWORD__OR_USERNAME_INVALID;
        }
        String baseDn = this.myConfiguration.getLdapBaseDn();
        String server = this.myConfiguration.getLdapServer();
        String domain = this.myConfiguration.getLdapDomain();
        try {
            this.connect(server, baseDn, String.valueOf(username) + "@" + domain, password);
        }
        catch (NamingException namingException) {
            return AuthenticationService.LDAPAuthenticationResult.PASSWORD_OR_USERNAME_INCORRECT;
        }
        return AuthenticationService.LDAPAuthenticationResult.AUTHENTICATED;
    }

    @Override
    public User createUser(X509Certificate certificate, int validityInDays) {
        Assertions.isDefined((Object)certificate, (String)ASSERTIONS_PARAMETER_NULL);
        return new CertificateUser(certificate, validityInDays);
    }

    @Override
    public User createUser(String userIdLdap, int validityInDays) {
        Assertions.isDefined((Object)userIdLdap, (String)ASSERTIONS_PARAMETER_NULL);
        return new LDAPUser(userIdLdap, validityInDays, this.myConfiguration.getLdapDomain());
    }

    @Override
    public X509Certificate loadCertificate(String file) throws AuthenticationException {
        Assertions.isDefined((Object)file, (String)"The parameter 'file' (path to the certificate) must not be null.");
        try {
            return CertUtil.loadCertificate((String)file);
        }
        catch (IOException e) {
            throw new AuthenticationException(e);
        }
        catch (GeneralSecurityException e) {
            throw new AuthenticationException(e);
        }
    }

    @Override
    public OpenSSLKey loadKey(String file) throws AuthenticationException {
        Assertions.isDefined((Object)file, (String)"The parameter 'file' (path to the key) must not be null.");
        try {
            return new BouncyCastleOpenSSLKey(file);
        }
        catch (IOException e) {
            throw new AuthenticationException(e);
        }
        catch (GeneralSecurityException e) {
            throw new AuthenticationException(e);
        }
    }

    @Override
    public User createUser(int validityInDays) {
        return new SingleUser(validityInDays);
    }

    private void connect(String server, String baseDn, String dn, String password) throws NamingException {
        Properties env = new Properties();
        env.setProperty("java.naming.factory.initial", CONTEXT_FACTORY_CLASS);
        env.setProperty("java.naming.provider.url", LDAP_PROTOCOL + server);
        env.setProperty("java.naming.security.authentication", LDAP_AUTH_METHOD);
        env.setProperty("java.naming.security.principal", dn);
        env.setProperty("java.naming.security.credentials", password);
        env.setProperty("java.naming.referral", REFERRAL);
        new InitialDirContext(env);
    }

    private boolean isPasswordNeeded(OpenSSLKey encryptedKey) {
        return encryptedKey.isEncrypted();
    }

    private boolean isPasswordCorrect(OpenSSLKey encryptedKey, String password) throws AuthenticationException {
        boolean correct = true;
        if (encryptedKey.isEncrypted()) {
            Assertions.isDefined((Object)password, (String)"If the key is encrypted the password must not be null.");
            try {
                encryptedKey.decrypt(password);
            }
            catch (BadPaddingException badPaddingException) {
                correct = false;
            }
            catch (RuntimeException e) {
                throw new AuthenticationException(e);
            }
            catch (InvalidKeyException e) {
                throw new AuthenticationException(e);
            }
            catch (GeneralSecurityException e) {
                throw new AuthenticationException(e);
            }
        }
        return correct;
    }

    private boolean isPrivateKeyBelongingToCertificate(X509Certificate certificate, OpenSSLKey encryptedKey) throws AuthenticationException {
        boolean belongs = true;
        PrivateKey privateKey = encryptedKey.getPrivateKey();
        PublicKey publicKey = certificate.getPublicKey();
        Random random = new Random();
        String original = Long.toString(Math.abs(random.nextLong()));
        try {
            Cipher cipher = Cipher.getInstance(privateKey.getAlgorithm());
            cipher.init(1, privateKey);
            byte[] encrypted = cipher.doFinal(original.getBytes());
            cipher.init(2, publicKey);
            byte[] decrypted = cipher.doFinal(encrypted);
            if (!original.equals(new String(decrypted))) {
                belongs = false;
            }
        }
        catch (BadPaddingException badPaddingException) {
            belongs = false;
        }
        catch (NoSuchAlgorithmException e) {
            throw new AuthenticationException(e);
        }
        catch (NoSuchPaddingException e) {
            throw new AuthenticationException(e);
        }
        catch (InvalidKeyException e) {
            throw new AuthenticationException(e);
        }
        catch (IllegalBlockSizeException e) {
            throw new AuthenticationException(e);
        }
        return belongs;
    }

    private boolean isSignedByTrustedCA(X509Certificate certificate) throws AuthenticationException {
        boolean signed = false;
        for (X509Certificate caCertificate : this.getCertificateAuthorities()) {
            try {
                caCertificate.checkValidity();
            }
            catch (CertificateNotYetValidException e) {
                throw new AuthenticationException(e);
            }
            catch (CertificateExpiredException e) {
                throw new AuthenticationException(e);
            }
            if (!certificate.getIssuerDN().equals(caCertificate.getSubjectDN())) continue;
            String encryptionAlgorithm = null;
            if (certificate.getSigAlgName().contains(RSA)) {
                encryptionAlgorithm = RSA;
            } else if (certificate.getSigAlgName().contains(DSA)) {
                encryptionAlgorithm = DSA;
            } else {
                throw new AuthenticationException("The encryption algorithm of the certificates's signature is not supported.");
            }
            try {
                Cipher cipher = Cipher.getInstance(encryptionAlgorithm);
                cipher.init(2, caCertificate.getPublicKey());
                cipher.doFinal(certificate.getSignature());
                signed = true;
            }
            catch (InvalidKeyException e) {
                throw new AuthenticationException(e);
            }
            catch (NoSuchAlgorithmException e) {
                throw new AuthenticationException(e);
            }
            catch (NoSuchPaddingException e) {
                throw new AuthenticationException(e);
            }
            catch (IllegalBlockSizeException e) {
                throw new AuthenticationException(e);
            }
            catch (BadPaddingException e) {
                throw new AuthenticationException(e);
            }
        }
        return signed;
    }

    private boolean isRevoced(X509Certificate certificate) throws AuthenticationException {
        boolean revoked = false;
        for (X509CRL revocationList : this.getCertificateRevocationLists()) {
            Date now = new Date();
            if (!revocationList.getIssuerDN().equals(certificate.getIssuerDN())) continue;
            if (revocationList.getThisUpdate().before(now) && (revocationList.getNextUpdate() == null || revocationList.getNextUpdate().after(now))) {
                if (!revocationList.isRevoked(certificate)) continue;
                revoked = true;
                continue;
            }
            throw new AuthenticationException("The CRL of the CA is not valid (e.g., it is expired).");
        }
        return revoked;
    }

    private List<X509Certificate> getCertificateAuthorities() {
        Vector<X509Certificate> certificates = new Vector<X509Certificate>();
        String absPath = null;
        for (String path : this.myConfiguration.getCaFiles()) {
            try {
                absPath = this.configurationService.resolveBundleConfigurationPath(this.bundleSymbolicName, path);
                certificates.add(CertUtil.loadCertificate((String)absPath));
            }
            catch (IOException iOException) {
                LOGGER.error((Object)StringUtils.format((String)CERTIFICATE_COULD_NOT_BE_LOADED, (Object[])new Object[]{absPath}));
            }
            catch (GeneralSecurityException generalSecurityException) {
                LOGGER.error((Object)StringUtils.format((String)CERTIFICATE_COULD_NOT_BE_LOADED, (Object[])new Object[]{absPath}));
            }
        }
        return certificates;
    }

    private List<X509CRL> getCertificateRevocationLists() {
        Vector<X509CRL> certificateRevocationLists = new Vector<X509CRL>();
        String absPath = null;
        for (String path : this.myConfiguration.getCrlFiles()) {
            try {
                absPath = this.configurationService.resolveBundleConfigurationPath(this.bundleSymbolicName, path);
                certificateRevocationLists.add(CertUtil.loadCrl((String)absPath));
            }
            catch (IOException iOException) {
                LOGGER.error((Object)StringUtils.format((String)CRL_COULD_NOT_BE_LOADED, (Object[])new Object[]{absPath}));
            }
            catch (GeneralSecurityException generalSecurityException) {
                LOGGER.error((Object)StringUtils.format((String)CRL_COULD_NOT_BE_LOADED, (Object[])new Object[]{absPath}));
            }
        }
        return certificateRevocationLists;
    }
}

