/*
 * Copyright 2006-2025 DLR, Germany
 * 
 * SPDX-License-Identifier: EPL-1.0
 * 
 * https://rcenvironment.de/
 */

package de.rcenvironment.core.configuration.logging;

import org.apache.logging.log4j.Level;
import org.apache.logging.log4j.core.Core;
import org.apache.logging.log4j.core.LogEvent;
import org.apache.logging.log4j.core.appender.rewrite.RewritePolicy;
import org.apache.logging.log4j.core.config.plugins.Plugin;
import org.apache.logging.log4j.core.config.plugins.PluginFactory;
import org.apache.logging.log4j.core.impl.Log4jLogEvent;
import org.apache.logging.log4j.message.FormattedMessage;
import org.apache.logging.log4j.message.Message;

/**
 * A custom log event filtering and rewriting policy for RCE. We primarily use this to mask irrelevant log events generated by libraries and
 * framework parts that can not be masked by simple package or class level rules.
 * 
 * Note that while such filtering rules MAY be used to suppress irrelevant warnings, or in unusual cases even errors, this is only
 * acceptable for root causes that cannot be fixed with reasonable effort instead AND that are estimated to cause no actual follow-up
 * problems. Such filtering rules MUST also be explained by commenting them in the filtering code below.
 * 
 * As logging is a relatively high-volume process, special care should be taken to implement all filtering rules as efficiently as possible.
 * 
 * @author Robert Mischke
 */
@Plugin(name = "RCELogEventRewritePolicy", elementType = "rewritePolicy", category = Core.CATEGORY_NAME)
public class RCELogEventRewritePolicy implements RewritePolicy {

    @PluginFactory
    public static RCELogEventRewritePolicy createInstance() {
        return new RCELogEventRewritePolicy();
    }

    @Override
    public LogEvent rewrite(LogEvent event) {

        // readability alias
        final Level originalLevel = event.getLevel();
        // when writing filters, prefer to only check the format string ("template" here) for efficiency
        final String messageTemplate = event.getMessage().getFormat();

        if (messageTemplate == null) {
            // unlikely, but make sure we do not cause NPEs within this filter
            return event;
        }

        // Rule for MANTIS-18211: Eliminate the highly verbose Felix SCR log output (about 35.500 lines per RCE run,
        // which bloats the RCE debug.log from <500 kB to >12 MB), and which we have not managed to disable at its
        // source so far. This filtering is complicated by the fact that Felix SCR uses the log (context) of each
        // bundle it operates on to log its messages. This mixes the Felix SCR log output with the actual application
        // log messages. It would still be much preferred to completely disable this logging for performance reasons.
        //
        // Attempts to filter this log output at the source included, but were not limited to:
        // - setting felix.log.level as a system property before OSGi startup
        // - setting org.osgi.service.log.admin.loglevel as a system property before OSGi startup
        // - adjusting the log levels for org.apache.felix.scr[.impl] via OSGi LoggerAdmin for the root LoggerContext
        //
        // TODO retest the SCR logging behavior after the next RCP upgrade
        if (originalLevel == Level.DEBUG && messageTemplate.startsWith("bundle ")) {
            // note that there is a theoretical risk of overfiltering by just checking for the "bundle " prefix;
            // however, a simple test for the additional presence of " : " does not match all SCR log lines
            // TODO consider adding a second, stricter regexp-based filter to rule out any false positives
            return suppressMessage(event);
        }

        // Rule for MANTIS-17266; a library warning which occurs fairly often, but with no clear path to prevent it
        if (originalLevel == Level.WARN && messageTemplate.endsWith("stream is already closed")) {
            return reduceLogLevelWithMessagePrefix(event, Level.DEBUG);
        }

        // Rule for MANTIS-18210: an error triggered by a NPE in Eclipse RCP code with no apparent way to prevent it
        if (originalLevel == Level.ERROR && messageTemplate.startsWith("FrameworkEvent ")) {
            final Throwable cause = event.getThrown();
            if (cause != null && cause.getMessage() != null) {
                if (cause.getMessage().startsWith("Exception in org.eclipse.debug.core.DebugPlugin.stop()")) {
                    return reduceLogLevelWithMessagePrefix(event, Level.DEBUG);
                }
            }
        }

        // Rule for some help index warnings that are part of the current Eclipse RCP (2023-03) being used, and
        // therefore out of our control. Note that these warnings seem to be remaining cases that were not fixed
        // as part of https://github.com/eclipse-platform/eclipse.platform.ua/issues/80 .
        if (originalLevel == Level.WARN && messageTemplate.startsWith("Unable to consume Lucene index from bundle 'org.eclipse.")) {
            return reduceLogLevelWithMessagePrefix(event, Level.DEBUG);
        }

        // return the unmodified event if no filter matched
        return event;
    }

    private Log4jLogEvent suppressMessage(LogEvent event) {
        // the log4j documentation states that returning "null" should work, but this actually causes problems,
        // so we reduce the level and let the default DEBUG level filter eliminate the event instead
        return reduceLogLevel(event, Level.TRACE);
    }

    private Log4jLogEvent reduceLogLevel(LogEvent event, Level newLevel) {
        return new Log4jLogEvent.Builder(event).setLevel(newLevel).build();
    }

    private Log4jLogEvent reduceLogLevelWithMessagePrefix(LogEvent event, Level newLevel) {
        final Message originalMessage = event.getMessage();
        final Message substituteMessage =
            new FormattedMessage(String.format("[Log level reduced from %s] %s",
                event.getLevel().name(), originalMessage.getFormat()), originalMessage.getParameters());
        return new Log4jLogEvent.Builder(event).setMessage(substituteMessage).setLevel(newLevel).build();
    }

}
