/*
 * Copyright 2006-2021 DLR, Germany
 * 
 * SPDX-License-Identifier: EPL-1.0
 * 
 * https://rcenvironment.de/
 */

package de.rcenvironment.core.component.wrapper.sandboxed;

import java.io.IOException;

import de.rcenvironment.core.utils.common.validation.ValidationFailureException;
import de.rcenvironment.core.utils.executor.CommandLineExecutor;

/**
 * An Interface that abstracts away all operations that depend on the selected delegation mode, for
 * example "local" or "SSH" (strategy pattern).
 * 
 * @author Robert Mischke
 */
public interface ExecutionEnvironment {

    /**
     * Set up the "static environment", ie everything that is not bound to a single invocation of
     * the target executable. A typical example would be a network connection to the execution host.
     * 
     * @throws IOException on I/O errors
     * @throws ValidationFailureException on invalid configuration values
     */
    void setupStaticEnvironment() throws IOException,
        ValidationFailureException;

    /**
     * Closes and/or cleans up the "static environment" created by a call to
     * {@link #setupStaticEnvironment()}.
     */
    void tearDownStaticEnvironment();

    /**
     * Creates a ready-to-use executor that has its work directory set to a new "sandbox".
     * Typipcally, this sandbox is simply a temporary directory on the execution host.
     * 
     * Note that this method always creates a new sandbox; this method is usually invoked by a
     * {@link SandboxBehaviour} implementation whenever a new sandbox directory is required.
     * 
     * @return a new {@link CommandLineExecutor} with a new associated sandbox
     * @throws IOException on I/O errors
     */
    CommandLineExecutor setupExecutorWithSandbox() throws IOException;

    /**
     * Disposes the sandbox associated with the given executor. The executor must have been
     * generated by the same {@link ExecutionEnvironment} instance, otherwise, the behaviour of this
     * method is undefined.
     * 
     * @param executor the {@link CommandLineExecutor} to gather sandbox information from
     * @throws IOException on I/O errors
     */
    void tearDownSandbox(CommandLineExecutor executor) throws IOException;

    /**
     * Generates a new, system-specific path in the target environment that can be used to
     * temporarily store arbitrary files. Usually a simple temporary directory, but with the two
     * properties that (1) it is empty on its creation, and (2) no other processes should write and
     * delete files to/from it.
     * 
     * @return the system-specific path of the storage directory
     * @throws IOException on I/O errors
     */
    String createUniqueTemporaryStoragePath() throws IOException;

}
